<?php
//-----------------------------------------------------------------
// Software: PHPCPR 
// Firm: PHPCPR.com / DirectSalesMLM.com / AutomaticWebSoftware.com
// Author: Jim Symonds Email: jim@directsalesmlm.com
// Copyright 2023-2024 Jim Symonds All Rights Reserved.
//-----------------------------------------------------------------

if (!isset($GLOBALS['debug'])) $GLOBALS['debug'] = false;

// Force PHP to report all errors
error_reporting(E_ALL); // Leave alone - will log all errors to error.log!
if (isset($GLOBALS['debug']) && $GLOBALS['debug'] === true) ini_set('display_errors', 1); // 1 means "on"
else ini_set('display_errors', 0); // 0 means "off"

if (!function_exists('custom_error_log')) {
    function custom_error_log($message) {
        // Fetching details from the backtrace to get file and line number
        $backtrace = debug_backtrace();
        $caller = isset($backtrace[1]) ? $backtrace[1] : null; // Gets the caller info

        $logFile = __DIR__ . "/error.log";
        $error_message = "[" . date("Y-m-d H:i:s") . "] ";
        if ($caller) {
            $error_message .= "Error in {$caller['file']} on line {$caller['line']}: ";
        }
        $error_message .= $message . PHP_EOL;

        // Append to the log file
        file_put_contents($logFile, $error_message, FILE_APPEND);

        // Optionally display the error message if debug is globally enabled
        if (isset($GLOBALS['debug']) && $GLOBALS['debug'] === true) {
            echo $message;
        }
    }
}


// Custom error handling function
if (!function_exists('customError')) {
	function customError($errno, $errstr, $errfile, $errline) {
		$logFile = fopen(__DIR__ . "/error.log", "a");
		$error_message = "[" . date("Y-m-d H:i:s") . "] Error: [$errno] $errstr - $errfile:$errline" . PHP_EOL;
		fwrite($logFile, $error_message);
		fclose($logFile);
		// Display the error if debugging is enabled
		if ($GLOBALS['debug'] === true) {
			echo debug_stmt('Error: '.$errno.', '.$errstr.' - '.$errfile.':, '.$errline);
		}
	}
}

// Set the error handler
set_error_handler("customError");
set_exception_handler(function($exception) {
    customError($exception->getCode(), $exception->getMessage(), $exception->getFile(), $exception->getLine());
});


// Register shutdown function to catch fatal errors
register_shutdown_function(function() {
	$last_error = error_get_last();
	if ($last_error && $last_error['type'] === E_ERROR) {
		// fatal error has occurred
		customError($last_error['type'], $last_error['message'], $last_error['file'], $last_error['line']);
	}
});


function create_vars($input) {
    if (is_array($input)) {
        // Input is an array: create multiple variables
        foreach ($input as $key => $value) {
            $GLOBALS[$key] = $value;
        }
        return array_keys($input);  // Return the names of the variables created
    } else if (is_string($input)) {
        // Input is a string: create a single variable
        $GLOBALS[$input] = $input;  // Create a global variable with the name passed
        return $input; // Return the name of the variable created		
    }
}

function show_array($array) {
    if ($GLOBALS['debug'] !== true) return null;
    $display = "";
    foreach ($array as $key => $value) {
        $display .= "<b>{$key}</b>: ";
        if (is_array($value)) {
            $display .= implode(', ', $value);
        } else {
            $display .= $value;
        }
        $display .= "<br>";		
        //$display .= "&nbsp;";
    }
    echo debug_stmt($display, 1);
}

function show_vars($vars) {
    if ($GLOBALS['debug'] !== true) return null;
    $display = ""; // Declare this variable here
    if (is_array($vars)) {
        foreach ($vars as $var_name) {
            if (isset($GLOBALS[$var_name])) {
                //$display .= "<br>";
                if (is_array($GLOBALS[$var_name])) {
                    $items = array();
                    foreach($GLOBALS[$var_name] as $key => $value) {
                        $items[] = "{$key} = '{$value}'";
                    }
                    $display .= implode(', ', $items);
                } else {
                    $display .= "{$var_name} = '{$GLOBALS[$var_name]}'";
                }
                $display .= "<br>";
            }
        }
    } else if (is_string($vars)) {
        $display .= "{$vars}: '{$GLOBALS[$vars]}'<br>";
    }
    echo debug_stmt($display, 1);
}

function getAssocValue($result, $key) {
    if (is_array($result) && isset($result[0][$key])) {
        return $result[0][$key];
    }
    return null;
}

function fetchAssocAll($result) {
    $buffer = [];

    if ($result instanceof mysqli_result) {
        if (strpos($result->query->queryString, '*') !== false) {
            while ($row = $result->fetch_assoc()) {
                $buffer[] = $row;
            }
        } else {
            if ($result->num_rows === 1) {
                $row = $result->fetch_assoc();
                return reset($row);
            } elseif ($result->num_rows === 0) {
                return null;
            } else {
                throw new Exception('Multiple rows returned for fetchAssocAll');
            }
        }
    } elseif ($result instanceof PDOStatement) {
        if (strpos($result->queryString, '*') !== false) {
            $buffer = $result->fetchAll(PDO::FETCH_ASSOC);
        } else {
            if ($result->rowCount() === 1) {
                $row = $result->fetch(PDO::FETCH_ASSOC);
                return reset($row);
            } elseif ($result->rowCount() === 0) {
                return null;
            } else {
                throw new Exception('Multiple rows returned for fetchAssocAll');
            }
        }
    } else {
        throw new Exception('Unsupported result type');
    }

    return $buffer;
}


function debug_stmt($debug_statement, $style=0) {
    if ($GLOBALS['debug'] !== true) return null;

    // Check if the statement is an array and convert it to a readable string
    if (is_array($debug_statement)) {
        $display = '<pre>' . print_r($debug_statement, true) . '</pre>';
    } else {
        $display = $debug_statement;
    }

    if ($style == 0) $alert_style = 'alert-danger';
    if ($style == 1) $alert_style = 'alert-primary';
    if ($style == 2) $alert_style = 'alert-warning'; 

    return '<div class="alert '.$alert_style.'" role="alert">'.$display.'</div>';    
}

function addslashesToItem($item) {
    // Check if the input is null or an empty string
    if ($item === null || $item === '') {
        return $item;
    }	
    if (is_array($item)) {
        // Flatten the array and apply addslashes to each element
        return array_map('addslashesToItem', $item);
    } else {
        // Apply addslashes to the item
        return addslashes((string)$item); // Cast to string to avoid any non-string values
    }
}


function debugQuery($query, $values) {
    if ($GLOBALS['debug'] !== true) return null;

    $debugQuery = $query;
    $offset = 0; // Keep track of the offset in the string
	foreach ($values as $value) {
		$placeholder = strpos($debugQuery, '?', $offset);
		if ($placeholder !== false) {
			// Determine the appropriate replacement for the placeholder
			if (is_numeric($value)) {
				// If the value is numeric, convert to integer or float
				$replacement = (int)$value == $value ? (int)$value : (float)$value;
			} elseif ($value instanceof DateTime) {
				// If the value is a DateTime object, format it as a string
				$replacement = "'" . $value->format('Y-m-d H:i:s') . "'";
			} elseif (is_array($value)) {
				// Flatten the array and handle each item
				$flattenedArray = new RecursiveIteratorIterator(new RecursiveArrayIterator($value));
				$processedItems = array_map('addslashesToItem', iterator_to_array($flattenedArray, false));
				$replacement = "'" . implode(', ', $processedItems) . "'";
			} else {
				// Check for null or empty string
				$replacement = $value === '' || $value === null ? "NULL" : "'" . addslashes($value) . "'";
			}
			
			// Replace the placeholder with the replacement, cast to string to be safe
			$replacementStr = (string)$replacement;
			$debugQuery = substr_replace($debugQuery, $replacementStr, $placeholder, 1);
			$offset = $placeholder + strlen($replacementStr);
		}
	}

    $debug_qry = 'DEBUG QRY: ' . $query; // display the prepared query with bound parameters (NO VALUES)
    $display = debug_stmt($debug_qry, 2);
    $debug_qry = 'DEBUG QRY: ' . $debugQuery;
    $display .= debug_stmt($debug_qry, 2);
    echo $display;
	//$_SESSION['debugQueryDisplay'] .= $display; 
}

function is_email($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
}


function isValidName($name) {
    // Check the length first
    if (strlen($name) > 50) {
        return false; // Name exceeds the maximum allowed length
    }

    // Check if the name contains only allowed characters including HTML entities
    return preg_match("/^[a-zA-Z-' &#0-9;]+$/", $name);
}


function isValidFullName($name) {
    // Check the length first
    if (strlen($name) > 100) {
        return false; // Name exceeds the maximum allowed length
    }

    // Check if the name contains only allowed characters including HTML entities
    return preg_match("/^[a-zA-Z-' &#0-9;]+$/", $name);
}


if (!function_exists('sanitize')) {
    function sanitize($data, $fieldName = '') {
		// Check if data is null and return it immediately
		if ($data === null) {
			return $data;
		}		
        // Recursive handling for arrays
        if (is_array($data)) {
            foreach ($data as $key => $value) {
                $data[$key] = sanitize($value, $key);
            }
            return $data;
        }

        // Check if the data is an email address
        if (filter_var($data, FILTER_VALIDATE_EMAIL)) {
            return filter_var($data, FILTER_SANITIZE_EMAIL);
        }
		
        // for URL sanitization
        if (filter_var($data, FILTER_VALIDATE_URL)) {
            return $data;
        }

        // Guess the type based on the field name
        $type = 'string'; // Default type
        if (strpos(strtolower($fieldName), 'email') !== false) {
            $type = 'email';
        } elseif (strpos(strtolower($fieldName), 'password') !== false) {
            // For password fields, return the data as is
            return $data;
        } elseif (is_numeric($data)) {
            $type = 'int';
        }

        // Apply different sanitization based on the type
        switch ($type) {
            case 'email':
                return filter_var($data, FILTER_SANITIZE_EMAIL);
            case 'int':
                //return filter_var($data, FILTER_SANITIZE_NUMBER_INT);
				return $data;
            case 'string':
            default:
				$fileExtensions = ['jpg', 'jpeg', 'gif', 'png'];
				$extension = strtolower(pathinfo($data, PATHINFO_EXTENSION));
				if (in_array($extension, $fileExtensions)) {
					// If the string is a filename with a specific extension, adjust the regex accordingly
					$data = preg_replace('/[^A-Za-z0-9 .\/:_-]/', '', $data);
                } else {
                    // For PHP 8.0 and later, perform custom sanitization
                    $data = strip_tags($data, '<b><i>'); // EXCLUDES <b><i> tags
                    // Adjust the regex to allow spaces, periods, colons, slashes, dashes, underscores, ?, =, @ [, ], linebreaks
					//$data = preg_replace('/[^A-Za-z0-9 .\/:\-_\?=]/', '', $data);
					//$data = preg_replace('/[^A-Za-z0-9 .\/:\-_\?=@\[\]\n\r]/', '', $data);
					//$data = preg_replace('/[^A-Za-z0-9 .\/:\-_\?=@\[\]\n\r,!\(\)\'*#&$<>]/', '', $data);
					$data = preg_replace('/[^A-Za-z0-9 .\/:\-_\?=@\[\]\n\r,!\(\)\'*#&$<>]/u', '', $data);

					/*Here is a list of all the characters that are explicitly allowed:

					Alphabets: Both uppercase and lowercase letters (A-Z, a-z).
					Numbers: All digits (0-9).
					Space: The space character.
					Special Characters:
					Period (.)
					Forward Slash (/)
					Colon (:)
					Hyphen (-)
					Underscore (_)
					Question Mark (?)
					Equal Sign (=)
					At Symbol (@)
					Left Square Bracket ([)
					Right Square Bracket (])
					Newline (\n)
					Carriage Return (\r)
					Comma (,)
					Exclamation Mark (!)
					Single Quote (')
					Left Parenthesis (()
					Right Parenthesis ())
					*/
                }
                return htmlspecialchars($data, ENT_QUOTES, 'UTF-8');
        }
    }
}



function conditionalTrim($value) {
    // Check if the value is a string to avoid operating on other data types
    if (is_string($value)) {
        // Count the number of words in the string
        $wordCount = str_word_count($value);

        // If the word count is less than X, trim the string
        if ($wordCount < 3) { // CONSIDERING AN ADDRESS MAY BE 3 WORDS
            return trim($value);
        }
        // Return the original string if word count is X or more
        return $value;
    }
    // Return the value as is if it's not a string
    return $value;
}


/*
// AUTO SANITIZE ALL GET AND POST VARS!
// WE KEEP THIS IN HERE, SO IT CANNOT BE FORGOTTEN OR EXCLUDED!!
// THIS BROKE THINGS, LIKE ADMIN PANEL URL REDIRECTS AND CC PAGE?
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    foreach ($_POST as $key => $value) {
        $_POST[$key] = sanitize($value, $key);
    }
    $_POST = array_map('trim', $_POST);
}

if (!empty($_GET)) {
    foreach ($_GET as $key => $value) {
        $_GET[$key] = sanitize($value, $key);
    }
    $_GET = array_map('trim', $_GET);
}*/


function validate_table_name($table) {
	//  checks if the table name only contains alphanumeric characters and underscores, the only characters allowed in MySQL table names.
    return preg_match('/^[a-zA-Z0-9_]+$/', $table);
}

function get_placeholder_type($value) {
    if ($value instanceof DateTime) {
        return 's';
    }
    switch (gettype($value)) {
        case 'string': return 's';
        case 'integer': return 'i';
        case 'double': return 'd';
        default: return 'b';
    }
}


function get_placeholder_type_pdo($value) {
    switch (gettype($value)) {
        case 'string': 
            return PDO::PARAM_STR;
        case 'integer': 
            return PDO::PARAM_INT;
        case 'double': 
            return PDO::PARAM_STR; // PDO doesn't have a separate constant for 'double', so we use PARAM_STR
        default: 
            return PDO::PARAM_LOB; // for 'blob' and other types, use PARAM_LOB
    }
}

// CREATE (INSERT)
function insert($table, $data, $live = 0, $show_debug = 0) {
    if ($GLOBALS['debug'] === true && $show_debug == 1) $show_debug = 1;
    else $show_debug = 0;
    
	if (!validate_table_name($table)) { // Validate table name
		customError(0,'Invalid table name', __FILE__, __LINE__);
        if ($show_debug == 1) {
            echo debug_stmt('No such table found!');
        } else {
            die('Unexpected event. Script terminated on line ' . __LINE__);
        }
        return false;
    }
	
	if (is_array($data)) {
        $fields = prepareFields($data);
        $placeholders = implode(', ', array_fill(0, count($data), '?'));
        $values = array_values($data);
    } else {
        $fields = prepareFields($table);
        $placeholders = '?';
        $values = $data;
    }

    $query = "INSERT INTO {$table} ({$fields}) VALUES ({$placeholders})";
	
	/*echo "Fields: {$fields}\n";
	echo "Placeholders: {$placeholders}\n";
	print_r($values);
	echo "Query: {$query}\n";
	exit;*/

    if ($show_debug == 1) debugQuery($query, $values);
	
	if ($live === 1) {
		// Choose database mode
		switch ($GLOBALS['db_mode']) {
			case 'pdo':
				return executePdo($query, $values, $show_debug);
			case 'mysqli':
				return executeMysqli($query, $values, $show_debug);
			default:
				throw new Exception("Invalid database mode: {$GLOBALS['db_mode']}");
		}
	} else {
		if ($show_debug == 1) echo debug_stmt('Insert Test Mode Only', 2); 
	}
	
}
// END INSERT

// INSERT SELECT
function insertSelect($insertIntoTable, $columns, $selectColumns, $fromTable, $additionalConditions = '', $live = 0, $show_debug = 0) {
    $insertColumns = implode(', ', $columns);
    $selectColumns = implode(', ', $selectColumns);

    $query = "INSERT INTO {$insertIntoTable} ({$insertColumns}) SELECT {$selectColumns} FROM {$fromTable} {$additionalConditions}";

    if ($show_debug == 1) {
        echo debug_stmt($query, 1);
    }

    if ($live === 1) {
        switch($GLOBALS['db_mode']) {
            case 'pdo':
                return executePdo($query, [], $show_debug);
            case 'mysqli':
                return executeMysqli($query, [], $show_debug);
            default:
                throw new Exception("Invalid database mode: {$GLOBALS['db_mode']}");
        }
    } else {
        if ($show_debug == 1) echo debug_stmt('Insert Select Test Mode Only', 2);
    }
} // END INSERT SELECT

/*
SELECT clause: List the fields you want to retrieve from the database.
FROM clause: Mention the primary table and any additional tables involved in the query (for joins).
WHERE clause: Specify the conditions to filter the rows.
GROUP BY clause: Group the results based on certain fields.
HAVING clause: Apply additional filters on the grouped results.
ORDER BY clause: Sort the results based on specified fields.
LIMIT clause: Limit the number of results returned.
*/
// READ (SELECT)
function select($selectFields, $mainTable, $joinConditions = [], $where = [], $groupBy = null, $having = null, $orderBy = null, $limit = 0, $show_debug = 0) {

    if ($GLOBALS['debug'] === true && $show_debug == 1) $show_debug = 1;
    else $show_debug = 0;

    // Prepare the GROUP BY clause if provided
    $groupByClause = '';
    if (isset($groupBy) && $groupBy !== null) {
        if (is_array($groupBy)) {
            $groupByClause = "GROUP BY " . implode(', ', $groupBy) . " ";
        } else {
            $groupByClause = "GROUP BY {$groupBy} ";
        }
    }

    // Prepare the HAVING clause if provided
    $havingClause = '';
    if (isset($having) && $having != null) {
        $havingClause = "HAVING {$having} ";
    }	

    // Prepare the ORDER BY clause if provided
    $orderByClause = '';
    if (isset($orderBy) && $orderBy !== null) {
        if(is_array($orderBy)) {
            $orderByClause = "ORDER BY " . implode(', ', $orderBy) . " ";
        } else {
            $orderByClause = "ORDER BY {$orderBy} ";
        }
    }

    // Check if main table is an array with 'table' and 'as' keys
    if (is_array($mainTable) && isset($mainTable['table'])) {
        $tableName = $mainTable['table'];
        $tableAlias = isset($mainTable['as']) ? $mainTable['as'] : $mainTable['table'];
    } else {
        $tableName = $mainTable;
        $tableAlias = $mainTable;
    }
	
	// Check if main table is an array with 'table' and 'as' keys
	// WORK IN PROGRESS
	/*if (is_array($mainTable)) {
		$tableParts = explode(' ', $mainTable[0]);
		$tableName = $tableParts[0];
		$tableAlias = isset($tableParts[2]) ? $tableParts[2] : $tableName;
	} else {
		// Use the mainTable string as both table name and alias
		$tableName = $mainTable;
		$tableAlias = $mainTable;
	}*/
	

    // Validate main table name
    /*if (!validate_table_name($tableName)) {
        if ($show_debug == 1) {
            echo debug_stmt('No such table found: ' . $tableName);
        } else {
            die('Unexpected event. Script terminated on line ' . __LINE__);
        }
        return false;
    }*/
	
    // Prepare query fields and where clause
    if ($selectFields === '*') {
        $fields = '*'; // Select all fields
    } else {
        $fields = prepareFields($selectFields);
    }	

	//print_r($where);
    list($whereClause, $whereValues, $whereTypes) = prepareWhereClause($where);
	//print_r([$whereClause, $whereValues, $whereTypes]); // add this line for debugging
	

    // Constructing join conditions
    $joinClause = '';
    if (!empty($joinConditions)) {
        foreach ($joinConditions as $condition) {
            if (is_array($condition) && isset($condition['table'])) {
                $joinTable = $condition['table'];
                $joinAlias = isset($condition['as']) ? $condition['as'] : $joinTable;
                $joinClause .= 'JOIN ' . $joinTable . ' AS ' . $joinAlias . ' ON ' . $condition['condition'] . ' ';
				
				if (!validate_table_name($condition['table'])) {
					if ($show_debug == 1) {
						echo debug_stmt('No such table found: ' . $condition['table']);
					} else {
						die('Unexpected event. Script terminated on line ' . __LINE__);
					}
					return false;
				}
            } else {
                // Handle invalid join condition format
				 $joinClause = '';
                if ($show_debug == 1) {
                    echo debug_stmt('Invalid join condition format');
                } /*else {
                    die('Unexpected event. Script terminated on line ' . __LINE__);
                }*/
                return false;
            }	
        }
    }
	
    // Construct the LIMIT clause - now supports offsets!
    $limitClause = '';
	if (is_array($limit)) {
		// If limit is an array with 'count' and optionally 'offset'
		$count = isset($limit['count']) ? intval($limit['count']) : 0;
		$offset = isset($limit['offset']) ? intval($limit['offset']) : 0;
		if ($count > 0) {
			$limitClause = "LIMIT {$offset}, {$count}";
		}
	} elseif (intval($limit) > 0) {
		// If limit is a straight number value
		$limitClause = "LIMIT " . intval($limit);
	}
	
	
    $query = "SELECT {$fields} FROM {$tableName} AS {$tableAlias} {$joinClause} WHERE {$whereClause} {$groupByClause} {$havingClause} {$orderByClause} {$limitClause}";
	
	//echo $query;
	//print_r($whereValues);
	//exit;

    if ($show_debug == 1) debugQuery($query, $whereValues);
	
	//exit;
	
	// Inside the select function, before executing the query
	/*echo "Final Query From Select: $query\n";
	echo "Where Values: ";
	print_r($whereValues);*/

    // Choose database mode
    switch($GLOBALS['db_mode']) {
        case 'pdo':
            return executePdo($query, $whereValues, $show_debug);
        case 'mysqli':
            return executeMysqli($query, $whereValues, $show_debug);
        default:
            throw new Exception("Invalid database mode: {$GLOBALS['db_mode']}");
    }
} // END SELECT


// UPDATE
// Correct usage of update function without join conditions
//$updateResult = update('your_table', $dataToUpdate, [], $whereConditions);
function update($mainTable, $data, $joinConditions = [], $where = [], $live = 0, $limit = 1, $show_debug = 0) {
    if ($GLOBALS['debug'] === true && $show_debug == 1) $show_debug = 1;
    else $show_debug = 0;
	
   // Check if main table is an array with 'table' and 'as' keys
    if (is_array($mainTable) && isset($mainTable['table'])) {
        $tableName = $mainTable['table'];
        $tableAlias = isset($mainTable['as']) ? $mainTable['as'] : $mainTable['table'];
    } else {
        $tableName = $mainTable;
        $tableAlias = $mainTable;
    }	

    /*if (!validate_table_name($table)) {
        if ($show_debug == 1) {
            echo debug_stmt('No such table found!');
        } else {
            die('Unexpected event. Script terminated on line ' . __LINE__);
        }
        return false;
    }*/

    $setFields = '';
    $bindValues = [];
    foreach ($data as $key => $value) {
        if (is_array($value) && isset($value['expression'])) {
            $setFields .= ($setFields == '' ? '' : ', ') . "$key = {$value['expression']}";
        } else if (is_numeric($value)) {
            $setFields .= ($setFields == '' ? '' : ', ') . "$key = $value";
        } else {
            $setFields .= ($setFields == '' ? '' : ', ') . "$key = ?";
            $bindValues[] = $value;
        }
    }

    /*$joinClause = '';
	if (isset($joinConditions) && is_array($joinConditions) && !empty($joinConditions)) {
        foreach ($joinConditions as $condition) {
            if (!validate_table_name($condition['table'])) {
                if ($show_debug == 1) {
                    echo debug_stmt('No such table found: ' . $condition['table']);
                } else {
                    die('Unexpected event. Script terminated on line ' . __LINE__);
                }
                return false;
            }

            $joinAlias = isset($condition['as']) ? $condition['as'] : $condition['table'];
            $joinClause .= 'JOIN ' . $condition['table'] . ' AS ' . $joinAlias . ' ON ' . $condition['condition'] . ' ';
        }
    }*/
	
    // Constructing join conditions
    $joinClause = '';
    if (!empty($joinConditions)) {
        foreach ($joinConditions as $condition) {
            if (is_array($condition) && isset($condition['table'])) {
                $joinTable = $condition['table'];
                $joinAlias = isset($condition['as']) ? $condition['as'] : $joinTable;
                $joinClause .= 'JOIN ' . $joinTable . ' AS ' . $joinAlias . ' ON ' . $condition['condition'] . ' ';
				
				if (!validate_table_name($condition['table'])) {
					if ($show_debug == 1) {
						echo debug_stmt('No such table found: ' . $condition['table']);
					} else {
						die('Unexpected event. Script terminated on line ' . __LINE__);
					}
					return false;
				}
            } else {
                // Handle invalid join condition format
				 $joinClause = '';
                if ($show_debug == 1) {
                    echo debug_stmt('Invalid join condition format');
                } /*else {
                    die('Unexpected event. Script terminated on line ' . __LINE__);
                }*/
                return false;
            }	
        }
    }	

    list($whereClause, $whereValues, $whereTypes) = prepareWhereClause($where);

    $limitClause = ''; 
	if (is_array($limit)) {
		// If limit is an array with 'count' and optionally 'offset'
		$count = isset($limit['count']) ? intval($limit['count']) : 0;
		$offset = isset($limit['offset']) ? intval($limit['offset']) : 0;
		if ($count > 0) {
			$limitClause = "LIMIT {$offset}, {$count}";
		}
	} elseif (intval($limit) > 0) {
		// If limit is a straight number value
		$limitClause = "LIMIT " . intval($limit);
	}
	

    $query = "UPDATE {$tableName} AS {$tableAlias} {$joinClause} SET {$setFields} WHERE {$whereClause} {$limitClause}";

    if ($show_debug == 1) debugQuery($query, array_merge($bindValues, $whereValues));

    if ($live === 1) {
        $allValues = array_merge($bindValues, $whereValues);

        switch ($GLOBALS['db_mode']) {
            case 'pdo':
                return executePdo($query, $allValues, $show_debug);
            case 'mysqli':
                return executeMysqli($query, $allValues, $show_debug);
            default:
                throw new Exception("Invalid database mode: {$GLOBALS['db_mode']}");
        }
    } else {
        if ($show_debug == 1) echo debug_stmt('Update Test Mode Only', 2); 
    }
} // END UPDATE

// DELETE
function delete($table, $joinConditions = [], $where = [], $orderBy = null, $live = 0, $limit = 1, $show_debug = 0) {
    // Check if debug mode is active
    if ($GLOBALS['debug'] === true && $show_debug == 1) $show_debug = 1;
    else $show_debug = 0;

    // Determine if an alias is being used
    $usingAlias = is_array($table) && isset($table['as']);
    $tableName = is_array($table) ? $table['table'] : $table;
    $tableAlias = $usingAlias ? " AS " . $table['as'] : '';

    // Constructing join conditions
    $joinClause = '';
    foreach ($joinConditions as $condition) {
        if (is_array($condition) && isset($condition['table'])) {
            $joinTable = $condition['table'];
            $joinAlias = isset($condition['as']) ? $condition['as'] : $joinTable;
            $joinClause .= ' JOIN ' . $joinTable . ' AS ' . $joinAlias . ' ON ' . $condition['condition'] . ' ';
        } else {
            // Handle invalid join condition format
            if ($show_debug == 1) {
                echo debug_stmt('Invalid join condition format');
            }
            return false;
        }
    }

    // Prepare WHERE clause
    list($whereClause, $whereValues, $whereTypes) = prepareWhereClause($where);

    // Prepare the ORDER BY clause if provided
    $orderByClause = '';
    if ($orderBy !== null && $orderBy !== '') {
        $orderByClause = "ORDER BY " . $orderBy . " ";
    }

    // Construct the LIMIT clause
    $limitClause = '';
    if (is_array($limit)) {
        $limitClause = "LIMIT " . $limit['count'];
        if (isset($limit['offset'])) {
            $limitClause .= " OFFSET " . $limit['offset'];
        }
    } elseif (intval($limit) > 0) {
        $limitClause = "LIMIT " . intval($limit);
    }

    // Check if join conditions exist and if LIMIT is used
    $usingJoinAndLimit = !empty($joinConditions) && !empty($limit);

    // Constructing DELETE statement
    $deleteTarget = $usingAlias ? $table['as'] : $tableName;
    if ($usingJoinAndLimit) {
        //$primaryKey = 'id'; // Replace 'id' with the actual primary key column name
        //$subQuery = "SELECT {$primaryKey} FROM {$tableName}{$tableAlias} {$joinClause} WHERE {$whereClause} {$orderByClause} {$limitClause}";
        $query = "DELETE {$deleteTarget} FROM {$tableName}{$tableAlias} WHERE {$whereClause} {$orderByClause} {$limitClause}";
    } else {
        $query = "DELETE FROM {$tableName}{$tableAlias} {$joinClause} WHERE {$whereClause} {$orderByClause} {$limitClause}";
    }

    // Debugging the query
    if ($show_debug == 1) debugQuery($query, $whereValues);

    // Execute the query
    if ($live === 1) {
        switch ($GLOBALS['db_mode']) {
            case 'pdo':
                return executePdo($query, $whereValues, $show_debug);
            case 'mysqli':
                return executeMysqli($query, $whereValues, $show_debug);
            default:
                throw new Exception("Invalid database mode: {$GLOBALS['db_mode']}");
        }
    } else {
        if ($show_debug == 1) echo debug_stmt('Delete Test Mode Only', 2);
    }
}





function deleteOLD($table, $joinConditions = [], $where = [], $orderBy = null, $live = 0, $limit = 1, $show_debug = 0) {
    if ($GLOBALS['debug'] === true && $show_debug == 1) $show_debug = 1;
    else $show_debug = 0;
	
    // Handle both array and string formats for table
    $tableName = is_array($table) ? $table['table'] : $table;
    $tableAlias = is_array($table) && isset($table['as']) ? " AS " . $table['as'] : '';

    // Constructing join conditions
    $joinClause = '';
    if (!empty($joinConditions)) {
        foreach ($joinConditions as $condition) {
            if (is_array($condition) && isset($condition['table'])) {
                $joinTable = $condition['table'];
                $joinAlias = isset($condition['as']) ? $condition['as'] : $joinTable;
                $joinClause .= 'JOIN ' . $joinTable . ' AS ' . $joinAlias . ' ON ' . $condition['condition'] . ' ';
				
				if (!validate_table_name($condition['table'])) {
					if ($show_debug == 1) {
						echo debug_stmt('No such table found: ' . $condition['table']);
					} else {
						die('Unexpected event. Script terminated on line ' . __LINE__);
					}
					return false;
				}
            } else {
                // Handle invalid join condition format
				 $joinClause = '';
                if ($show_debug == 1) {
                    echo debug_stmt('Invalid join condition format');
                } /*else {
                    die('Unexpected event. Script terminated on line ' . __LINE__);
                }*/
                return false;
            }	
        }
    }
	
    list($whereClause, $whereValues, $whereTypes) = prepareWhereClause($where);
	
    // Prepare the ORDER BY clause if provided
    $orderByClause = '';
    if (isset($orderBy) && $orderBy !== null) {
        if(is_array($orderBy)) {
            $orderByClause = "ORDER BY " . implode(', ', $orderBy) . " ";
        } else {
            $orderByClause = "ORDER BY {$orderBy} ";
        }
    }

    // Construct the LIMIT clause - now supports offsets!
    $limitClause = '';
	if (is_array($limit)) {
		// If limit is an array with 'count' and optionally 'offset'
		$count = isset($limit['count']) ? intval($limit['count']) : 0;
		$offset = isset($limit['offset']) ? intval($limit['offset']) : 0;
		if ($count > 0) {
			$limitClause = "LIMIT {$offset}, {$count}";
		}
	} elseif (intval($limit) > 0) {
		// If limit is a straight number value
		$limitClause = "LIMIT " . intval($limit);
	}	

    //$query = "DELETE FROM {$table} {$joinClause} WHERE {$whereClause} {$orderByClause} {$limitClause}";
	    // Construct the DELETE query
    $query = "DELETE " . (is_array($table) ? $table['as'] : $tableName) . 
             " FROM {$tableName}{$tableAlias} {$joinClause} WHERE {$whereClause} {$orderByClause} {$limitClause}";

    if ($show_debug == 1) debugQuery($query, $whereValues);

    if ($live === 1) {
        // Choose database mode
        switch ($GLOBALS['db_mode']) {
            case 'pdo':
                return executePdo($query, $whereValues, $show_debug);
            case 'mysqli':
                return executeMysqli($query, $whereValues, $show_debug);
            default:
                throw new Exception("Invalid database mode: {$GLOBALS['db_mode']}");
        }
    } else {
        if ($show_debug == 1) echo debug_stmt('Delete Test Mode Only', 2); 
    }   
}

// TRUNCATE
function truncateTable($table, $live = 0, $show_debug = 0) {
    if ($GLOBALS['debug'] === true && $show_debug == 1) $show_debug = 1;
    else $show_debug = 0;

    if (!validate_table_name($table)) {
        if ($show_debug == 1) {
            echo debug_stmt('No such table found!');
        } else {
            die('Unexpected event. Script terminated on line ' . __LINE__);
        }
        return false;
    }

    $query = "TRUNCATE TABLE {$table}";

    if ($show_debug == 1) debugQuery($query, []);

    if ($live === 1) {
        // Choose database mode
        switch ($GLOBALS['db_mode']) {
            case 'pdo':
                return executePdo($query, [], $show_debug);
            case 'mysqli':
                return executeMysqli($query, [], $show_debug);
            default:
                throw new Exception("Invalid database mode: {$GLOBALS['db_mode']}");
        }
    } else {
        if ($show_debug == 1) echo debug_stmt('Truncate Test Mode Only', 2);
    }
}
 

function prepareFields($values) {
    if ($values === '*') {
        return '*';
    } elseif (is_array($values)) {
        if (array_keys($values) !== range(0, count($values) - 1)) {
            // Associative array
            return implode(', ', array_keys($values));
        } else {
            // Numeric array
            return implode(', ', $values);
        }
    } else {
        return $values;
    }
}


function prepareWhereClause($where) {
    if ($where === ['1']) {
        return ['1', [], '']; // Always true condition
    }

    list($clause, $values, $types) = parseConditions($where);
    return [implode(' AND ', $clause), $values, $types];
}

function parseConditions($conditions, $parentOperator = 'AND') {
    if (!is_array($conditions)) {
        return [[], [], ''];
    }

    $whereClauses = [];
    $whereValues = [];
    $whereTypes = '';

    foreach ($conditions as $key => $val) {
        if ($key === 'AND' || $key === 'OR') {
            // Handle nested AND/OR conditions
            $nestedClauses = [];
            foreach ($val as $nestedKey => $nestedVal) {
                if (is_array($nestedVal) && is_numeric($nestedKey)) {
                    // Handle multiple conditions for different fields
                    foreach ($nestedVal as $field => $condition) {
                        $result = processCondition($field, $condition);
                        $nestedClauses[] = $result['clause'];
                        $whereValues = array_merge($whereValues, $result['values']);
                        $whereTypes .= $result['types'];
                    }
                } else {
                    // Handle direct nested condition
                    $result = processCondition($nestedKey, $nestedVal);
                    $nestedClauses[] = $result['clause'];
                    $whereValues = array_merge($whereValues, $result['values']);
                    $whereTypes .= $result['types'];
                }
            }
            $whereClauses[] = '(' . implode(" $key ", $nestedClauses) . ')';
        } else if (is_array($val) && !empty($val)) {
            if (is_array(current($val))) {
                // Adjusted to handle multiple conditions for the same field as AND/OR
                $multiConditions = [];
                foreach ($val as $condition) {
                    $result = processCondition($key, $condition);
                    $multiConditions[] = $result['clause'];
                    $whereValues = array_merge($whereValues, $result['values']);
                    $whereTypes .= $result['types'];
                }
                $whereClauses[] = '(' . implode(" AND ", $multiConditions) . ')';
            } else {
                // Direct condition with single array as value
                $result = processCondition($key, $val);
                $whereClauses[] = $result['clause'];
                $whereValues = array_merge($whereValues, $result['values']);
                $whereTypes .= $result['types'];
            }
        } else {
            // Direct conditions
            $result = processCondition($key, $val);
            $whereClauses[] = $result['clause'];
            $whereValues = array_merge($whereValues, $result['values']);
            $whereTypes .= $result['types'];
        }
    }

    return [$whereClauses, $whereValues, $whereTypes];
}

function processCondition($field, $val) {
    $clause = '';
    $values = [];
    $types = '';

    if (is_array($val) && count($val) === 2 && !is_numeric($field)) {
        list($operator, $v) = $val;
        $operator = strtoupper($operator);

        switch ($operator) {
            case 'IN':
            case 'NOT IN':
                if (is_array($v)) {
                    $placeholders = implode(', ', array_fill(0, count($v), '?'));
                    $clause = "$field $operator ($placeholders)";
                    foreach ($v as $item) {
                        $values[] = $item;
                        $types .= get_placeholder_type($item);
                    }
                }
                break;
            
            case 'IS NULL':
            case 'IS NOT NULL':
                $clause = "$field $operator";
                break;
            default:
                $clause = "$field $operator ?";
                $values[] = $v;
                $types .= get_placeholder_type($v);
                break;
        }
    } else if (is_array($val) && count($val) === 1 && is_numeric($field)) {
        $field = key($val);
        $v = current($val);

        $clause = "$field = ?";
        $values[] = $v;
        $types .= get_placeholder_type($v);       
    } else { // Direct condition (implied equality)
        $clause = "$field = ?";
        $values[] = $val;
        $types .= get_placeholder_type($val);
    }

    return ['clause' => $clause, 'values' => $values, 'types' => $types];
}



function processOrConditions($field, $conditions) {
    $clauseParts = [];
    $values = [];
    $types = '';

    foreach ($conditions as $condition) {
        $result = processCondition($field, $condition);
        $clauseParts[] = $result['clause'];
        $values = array_merge($values, $result['values']);
        $types .= $result['types'];
    }

    return [
        'clause' => '(' . implode(' OR ', $clauseParts) . ')',
        'values' => $values,
        'types' => $types
    ];
}


function is_datetime($value) {
    if ($value === '') return false; // Check for empty string
    try {
        new DateTime($value);
        return true;
    } catch (Exception $e) {
        return false;
    }
}

 
/**
* Performs database operations for SELECT, UPDATE, INSERT, and DELETE queries using PDO.
*
* @param string $query The query to be executed.
* @param array $values The array of values to be bound to the query.
* @param int $show_debug Determines whether to enable debug mode or not.
* @return mixed The result set for SELECT query, the number of affected rows for UPDATE, INSERT, and DELETE queries.
*/

function executePdo($query, $values, $show_debug) {
    if ($GLOBALS['debug'] === true && $show_debug == 1) $show_debug = 1;
    else $show_debug = 0;
	
    global $db_conn;  // Assuming $db_conn is your pdo instance
    try {
        $stmt = $db_conn->prepare($query);

        // Bind parameters
        foreach ($values as $key => &$value) {
            $stmt->bindValue($key+1, $value, get_placeholder_type_pdo($value));
        }		

        $stmt->execute();

        $queryType = strtoupper(substr(trim($query), 0, 6));
		if ($show_debug == 1) {
			echo debug_stmt($queryType.' DB command executed!', 1);
			//$_SESSION['debugQueryDisplay'] .= debug_stmt($queryType.' DB command executed!', 1);
		}		
        if ($queryType === 'INSERT') {
            $lastInsertedId = $db_conn->lastInsertId();
            return array('affected_rows' => $stmt->rowCount(), 'last_insert_id' => $lastInsertedId);
        } elseif ($queryType === 'SELECT') {
            return $stmt->fetchAll(PDO::FETCH_ASSOC);  
        } else {
            return $stmt->rowCount();  
        }		

    } catch (PDOException $e) {
        customError($e->getCode(), $e->getMessage(), __FILE__, __LINE__);
        if ($show_debug == 1) {
            echo debug_stmt('Error in PDO execution: ' . $e->getMessage());
        } else {
            die('Unexpected event. Script terminated on line ' . __LINE__);
        }
        return false;
    } finally {
        //$stmt = null;       
        //$db_conn = null;        
    }
}


/**
* Performs database operations for SELECT, UPDATE, INSERT, and DELETE queries using MySQLi.
*
* @param string $query The query to be executed.
* @param array $values The array of values to be bound to the query.
* @param int $show_debug Determines whether to enable debug mode or not.
* @return mixed The result set for SELECT query, the number of affected rows for UPDATE, INSERT, and DELETE queries.
*/

function executeMysqli($query, $whereValues, $show_debug) {
    if ($GLOBALS['debug'] === true && $show_debug == 1) $show_debug = 1;
    else $show_debug = 0;
	
    global $db_conn;  // Assuming $db_conn is your mysqli instance
	
	if ($db_conn === null) {
		die('Database connection ($db_conn) is not established.');
	}
	
	// Check if the query is a TRUNCATE TABLE statement
	if (strtoupper(substr(trim($query), 0, 8)) === 'TRUNCATE') {
		// Execute the TRUNCATE TABLE query
		$execute = mysqli_query($db_conn, $query);
		if ($execute === false) {
			customError(mysqli_errno($db_conn), mysqli_error($db_conn), __FILE__, __LINE__);
			if ($show_debug == 1) {
				echo debug_stmt('Error executing TRUNCATE TABLE statement with mysqli: ' . mysqli_error($db_conn));
			} else {
				die('Unexpected event. Script terminated on line ' . __LINE__);
			}
			return false;
		}
		if ($show_debug == 1) {
			echo debug_stmt('TRUNCATE TABLE command executed!', 1);
		}
		return true;
	}

    $stmt = mysqli_prepare($db_conn, $query);
    if ($stmt === false) {
        customError(mysqli_errno($db_conn), mysqli_error($db_conn), __FILE__, __LINE__);
        if ($show_debug == 1) {
            echo debug_stmt('Error preparing statement with mysqli: ' . mysqli_error($db_conn));
        } else {
            die('Unexpected event. Script terminated on line ' . __LINE__);
        }
        return false;
    }

    /*// Bind parameters
    $types = '';
    $bindValues = [];
    foreach ($whereValues as $value) {
        $types .= get_placeholder_type($value);
        $bindValues[] = $value;
    }
    mysqli_stmt_bind_param($stmt, $types, ...$bindValues);
    array_unshift($bindValues, $stmt, $types);
    call_user_func_array('mysqli_stmt_bind_param', $bindValues);	
    //call_user_func_array('mysqli_stmt_bind_param', array_merge([$stmt, $types], $bindValues));*/
	
	// Bind parameters
/*	$types = '';
	$bindValues = [];
	foreach ($whereValues as $value) {
		$types .= get_placeholder_type($value);
		$bindValues[] = &$value;
	}
	array_unshift($bindValues, $types);
	
	echo "Types: {$types}\n";
	print_r($bindValues);

	call_user_func_array([$stmt, 'bind_param'], $bindValues);*/
	
    // Check if there are parameters to bind
    if (!empty($whereValues)) {
        $types = '';
        $bindValues = [$types];
        foreach ($whereValues as $key => $value) {
            $types .= get_placeholder_type($value);
            $bindValues[] = &$whereValues[$key];
        }
        $bindValues[0] = $types;

        call_user_func_array([$stmt, 'bind_param'], $bindValues);
    }

    // Execute the prepared statement
    $execute = mysqli_stmt_execute($stmt);
    if ($execute === false) {
        customError(mysqli_errno($db_conn), mysqli_error($db_conn), __FILE__, __LINE__);
        if ($show_debug == 1) {
            echo debug_stmt('Error executing statement with mysqli: ' . mysqli_error($db_conn));
        } else {
            die('Unexpected event. Script terminated on line ' . __LINE__);
        }
        return false;
    } else {
		$qry_type = strtoupper(substr($query, 0, 6));
		//if ($show_debug == 1) echo debug_stmt($qry_type.' DB command executed!', 1);
	}	

    // Retrieve the last inserted ID if it's an INSERT query
    $queryType = strtoupper(substr(trim($query), 0, 6));
	
	$result = null; // Initialize $result
	
    if ($queryType === 'INSERT') {
		$lastInsertedId = mysqli_insert_id($db_conn);
		$aff_rows = array('affected_rows' => mysqli_stmt_affected_rows($stmt), 'last_insert_id' => $lastInsertedId);
		if ($show_debug == 1) echo debug_stmt($qry_type.' DB command executed! '.print_r($aff_rows, true), 1);

		return array('affected_rows' => mysqli_stmt_affected_rows($stmt), 'last_insert_id' => $lastInsertedId);
	} else if ($queryType === 'UPDATE' || $queryType === 'DELETE') {
		$aff_rows = array('affected_rows' => mysqli_stmt_affected_rows($stmt));
		if ($show_debug == 1) echo debug_stmt($qry_type.' DB command executed! '.print_r($aff_rows, true), 1);

		
		return array('affected_rows' => mysqli_stmt_affected_rows($stmt));
	} else if ($queryType === 'SELECT') {
        // Retrieve the result set
        $result = mysqli_stmt_get_result($stmt); // Get result set for SELECT query
		$rows = mysqli_num_rows($result);
		if ($show_debug == 1) echo debug_stmt($qry_type.' DB command executed! Row count: '.$rows, 1);
		

        // Fetch the result set
        $vars_array = [];
        while ($row = mysqli_fetch_assoc($result)) {
            $vars_array[] = $row;
        }

		/*return array(
			//'affected_rows' => mysqli_stmt_affected_rows($stmt),
			'vars_array' => $vars_array
		);*/
		return $vars_array;
    }
    
    mysqli_stmt_close($stmt);
    // If $result is set, then we have a result set to free
    if (isset($result)) {
        mysqli_free_result($result);
    }
	//mysqli_close($db_conn);
}

// Check if the current page's filepath contains 'admin'
if (strpos($_SERVER['REQUEST_URI'], 'admin') !== false) {
    // If the request method is POST, sanitize and conditionally trim all values
    if ($_SERVER['REQUEST_METHOD'] == 'POST') {
        foreach ($_POST as $key => $value) {
            $_POST[$key] = sanitize($value, $key);
        }
        $_POST = array_map('conditionalTrim', $_POST);
    }

    // Sanitize and conditionally trim all GET values
    if (!empty($_GET)) {
        foreach ($_GET as $key => $value) {
            $_GET[$key] = sanitize($value, $key);
        }
        $_GET = array_map('conditionalTrim', $_GET);
    }
}

// COMMENT THIS OUT WHEN TIME ALLOWS, TO CATCH THE SMALL WARNINGS TO FIX!
// UNCOMMENT TO KEEP WARNINGS DOWN TO ONLY CRITICAL STUFF!
restore_error_handler();
restore_exception_handler();
?>